package cz.drg.clasificator.readers;

import cz.drg.clasificator.args.argevaluation.ConfigurableIOClass;
import cz.drg.clasificator.exception.ShutdownException;
import cz.drg.clasificator.setting.Settings;
import cz.drg.clasificator.setting.program.IOclass;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.util.FilesCache;
import cz.drg.clasificator.util.HeaderList;
import cz.drg.clasificator.util.ResultSetToHeaderListConvertor;
import cz.drg.clasificator.writers.CsvWriter;
import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.logging.Level;
import java.util.logging.Logger;
import static cz.drg.clasificator.util.OutputHelper.*;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.zip.ZipFile;


public class DBReader implements InputReader, ConfigurableIOClass{

    
    private static final String TEMP_FILE_NAME = "./log/DRGtemp%d.csv";
    
    private boolean isInitialized = false;
    private IOclass ioClass;
    private ZipFile pmmlInput;
    private Connection connection;
    private File tempCsv;
    private CsvReader tempCsvReader;
    
    public DBReader() {
        try {
            pmmlInput = new ZipFile(Settings.getProgramSettings().getDefaultPmmlPath());
        } 
        catch (IOException ex) {
            dualLog("Error loading default PMML zip file: '"+Settings.getProgramSettings().getDefaultPmmlPath()+"'");
            System.exit(-1);
        }
        
    }
    
    private final String SELECT_ALL_QUERY = "SELECT * FROM %s";
    private final String SELECT_BY_ID_QUERY = SELECT_ALL_QUERY+" WHERE id_pripadu LIKE '%s' OR CAST(run_id AS VARCHAR(50)) LIKE '%s'";
    
    private String getSelectByCaseIdOrRunIdQuery(String id){
        return String.format(SELECT_BY_ID_QUERY, ioClass.getDatabaseTargetTable(), id, id);
    }
    
    private String getSelectAllQuery(){
        return String.format(SELECT_ALL_QUERY, ioClass.getDatabaseTargetTable());
    }
    
    @Override
    public void initialize(){
        init(getSelectAllQuery());
    }
    
    public void initialize(String id){
        init(getSelectByCaseIdOrRunIdQuery(id));
    }
    
    private void init(String query){
        close();
        
        if(!isConnected()){
            connection = createNewConnection();
        }
        
        try {
            
            PreparedStatement prepareStatement = connection.prepareStatement(query);
            ResultSet dbInput = prepareStatement.executeQuery();
            
            String defaultCharset = Settings.getProgramSettings().getDefaultCharset();
            
            tempCsv = createTempCsv();
            CsvWriter writer = new CsvWriter(tempCsv, "\t", defaultCharset);

            writeRowsFromResultSet(dbInput, writer);
            
            dbInput.close();
            prepareStatement.close();
            
        } catch (SQLException ex) {
            dualLog(ex.getMessage());
            throw new ShutdownException(Constants.ERR_SQL_READ);
        } 
        finally{
            try {
                connection.close();
            } catch (SQLException ex) {
                Logger.getLogger(DBReader.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
        tempCsvReader = new CsvReader(tempCsv);
        isInitialized = true;
    }
    
    @Override
    public ZipFile readPmmlInput() {
        return pmmlInput;
    }

    @Override
    public HeaderList readNextEntryBatch() {
        
        if(!isInitialized){
            initialize();
        }
        
        if(tempCsvReader.hasNextEntryBatch()){
            
            return tempCsvReader.readNextEntryBatch();
        }
            
        throw new ShutdownException(Constants.ERR_READ_NEXT_BATCH_DB);
    }
    
    private File createTempCsv(){
        
        
        File tempFile = null;
        try {
            String filePath = String.format(TEMP_FILE_NAME, System.currentTimeMillis());
            
            errLog("Creating temporary csv file '"+filePath+"'.");
            
            tempFile = new File(filePath);
            tempFile.createNewFile();
            
            FilesCache.addCreatedFile(tempFile.getPath());
            
            errLog("Temporary file created at "+tempFile.getPath()+"'.");
            
        } catch (IOException ex) {
            dualLog(ex.getMessage());
            throw new ShutdownException(Constants.ERR_CREATE_TEMP_CSV);
        }
        
        return tempFile;
    }
    
    private Connection createNewConnection(){
        
        String databaseUrl = getTargetDatabaseUrl();
        
        errLog("Creating new connection to database '"+databaseUrl+"' for reading.");
        
        String username = ioClass.getDatabaseUser();
        String password = ioClass.getDatabasePassword();
        
        if(username == null || password == null){
            Scanner scan = new Scanner(System.in);
            dualLog("Enter username:");
            username = scan.nextLine();
            dualLog("Enter password:");
            password = scan.nextLine();
        }
        
        Connection connection = null;
        
        try {
                   
            connection = DriverManager.getConnection(databaseUrl, username, password);
            
        } catch (SQLException ex) {
            dualLog(ex.getMessage());
            throw new ShutdownException(String.format(Constants.ERR_SQL_CONNECTION, databaseUrl));
        } 
        
        return connection;
    }
    
    private List<String> readHeadersFromResultSet(ResultSet resultSet){
        
        List<String> headerList = null;
        
        try {
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int columnCount = rsmd.getColumnCount();

            //HashMap headers = new HashMap<>();
            headerList = new ArrayList<>();
            
            for (int i = 1; i <= columnCount; i++ ) {
                String name = rsmd.getColumnName(i);
                //headers.put(name, i-1);
                headerList.add(name);
            }
            
        } catch (SQLException ex) {
            Logger.getLogger(ResultSetToHeaderListConvertor.class.getName()).log(Level.SEVERE, null, ex);
        }
        
        return headerList;
    }
    
    private void writeRowsFromResultSet(ResultSet resultSet, CsvWriter writer){
        
        try {
            List<String> headersFromResultSet = readHeadersFromResultSet(resultSet);
            String headerRow = listToString(headersFromResultSet);
            
            
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int columnCount = rsmd.getColumnCount();
            
            int batchSize = Settings.getProgramSettings().getEvaluationBatchSize();
            List<String> rowColumns = new ArrayList<>();
            List<String> rows = new ArrayList<>();
            rows.add(headerRow);
            
            long counter = 1;
            ResultSetMetaData metaData = resultSet.getMetaData();
            
            while(resultSet.next()){
                
                for (int i = 1; i <= columnCount; i++) {
                    
                    String columnName = metaData.getColumnName(i);
                    
                    String colVal = resultSet.getString(i);
                    
                    if(colVal == null){
                        colVal = "";
                    }
                    else if(columnName.equalsIgnoreCase("datum_pri") || columnName.equalsIgnoreCase("datum_pro")){
                        colVal = colVal.replaceAll("-", "");
                        colVal = colVal.substring(0, 8);
                    }
                    rowColumns.add(colVal);
                }
                
                rows.add(listToString(rowColumns));
                rowColumns.clear();
                
                if(counter % batchSize == 0){
                    writer.writeOutput(rows);
                    rows.clear();
                    //csv reader expects the first row to be header row in each batch
                    rows.add(headerRow);
                }
                
                counter++;
            }
            writer.writeOutput(rows);
            rows.clear();
            
        } catch (SQLException ex) {
            Logger.getLogger(ResultSetToHeaderListConvertor.class.getName()).log(Level.SEVERE, null, ex);
        }
        
    }
    
    private String listToString(List<String> columns){
        
        String result = "";
        
        for (int i = 0; i < columns.size(); i++) {
            String col = columns.get(i);
            
            result += col;
            
            //last one
            if(i != columns.size()-1){
                result += "\t";
            }
        }
        
        return result;
    }
    
    @Override
    public boolean hasNextEntryBatch() {
        return tempCsvReader.hasNextEntryBatch();
    }
    
    @Override
    public void setIoClass(IOclass config) {
        this.ioClass = config;
    }

    private String getTargetDatabaseUrl(){
        return ioClass.getDatabaseConnection();
    }
    
    private boolean isConnected(){
        try {
            return connection != null && !connection.isClosed();
        } catch (SQLException ex) {
            Logger.getLogger(DBReader.class.getName()).log(Level.SEVERE, null, ex);
        }
        return false;
    }

    @Override
    public void close() {
        if(tempCsv == null) return;
        
        try {
            tempCsvReader.close();
            
            Files.deleteIfExists(tempCsv.toPath());
            FilesCache.freeRemovedFile(tempCsv.getPath());
        } catch (IOException ex) {
            Logger.getLogger(DBReader.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
    
}
